using System;
using System.IO;
using System.Text;
using System.Windows.Forms;

using KeePass.Plugins;
using KeePassLib;
using KeePassLib.Security;

namespace KeePass2JSON
{
	public sealed class KeePass2JSONExt : Plugin
	{
		private IPluginHost p_host = null;
		private ToolStripSeparator m_separator = null;
		private ToolStripMenuItem m_export = null;
		private bool first_entry;

		public override bool Initialize(IPluginHost host)
		{
			p_host = host;

			m_separator = new ToolStripSeparator();
			m_export = new ToolStripMenuItem();
			m_export.Text = "Export to JSON...";
			m_export.Click += OnExport;

			ToolStripItemCollection menu = p_host.MainWindow.ToolsMenu.DropDownItems;
			menu.Add(m_separator);
			menu.Add(m_export);

			return true;
		}

		public override void Terminate()
		{
			ToolStripItemCollection menu = p_host.MainWindow.ToolsMenu.DropDownItems;
			menu.Remove(m_separator);
			menu.Remove(m_export);
		}

		private void OnExport(Object sender, EventArgs e)
		{
			if (!p_host.Database.IsOpen) {
				MessageBox.Show("Database needs to be open", "JSON plugin");
				return;
			}

			SaveFileDialog dialog = new SaveFileDialog();
			dialog.Filter = "JavaScript files (*.js)|*.js|All files (*.*)|*.*" ;
			dialog.FilterIndex = 1;
			dialog.AddExtension = true;
			dialog.RestoreDirectory = true;

			if (dialog.ShowDialog() == DialogResult.OK) {
				Stream fs = dialog.OpenFile();
				if (fs != null) {
					StreamWriter sw = new StreamWriter(fs);
					sw.Write("[");
					first_entry = true;
					p_host.Database.RootGroup.TraverseTree(TraversalMethod.PreOrder, null, entry => export(entry, sw));
					sw.Write("\n]\n");
					sw.Close();
					fs.Close();
				}
			}
		}

		private bool export(PwEntry entry, StreamWriter sw)
		{
			if (entry.ParentGroup.Uuid == p_host.Database.RecycleBinUuid) return false;

			StringBuilder s = new StringBuilder(128);
			if (!first_entry) s.Append(',');
			s.Append("\n\t{\n");
			s.Append("\t\t\"group\": "      ); s.Append(quoted_string(entry.ParentGroup.Name));
			s.Append(",\n\t\t\"title\": "   ); s.Append(quoted_string(entry, "Title"   ));
			s.Append(",\n\t\t\"username\": "); s.Append(quoted_string(entry, "UserName"));
			s.Append(",\n\t\t\"password\": "); s.Append(quoted_string(entry, "Password"));
			s.Append(",\n\t\t\"url\": "     ); s.Append(quoted_string(entry, "URL"     ));
			s.Append(",\n\t\t\"notes\": "   ); s.Append(quoted_string(entry, "Notes"   ));
			if (entry.Expires) {
				Int64 t = (Int64)entry.ExpiryTime.ToUniversalTime().Subtract(new DateTime(1970, 1, 1)).TotalSeconds;
				s.Append(",\n\t\t\"expires\": " ); s.Append(t);
			}
			s.Append("\n\t}");
			sw.Write(s);

			first_entry = false;
			return true;
		}

		private static string quoted_string(PwEntry entry, string name)
		{
			ProtectedString ps = entry.Strings.Get(name);
			return (ps == null) ? @"""""" : quoted_string(ps.ReadString());
		}

		private static string quoted_string(string s)
		{
			StringBuilder es = new StringBuilder(@"""", s.Length + 20);

			foreach (char c in s) {
				switch (c) {
					case '"' : es.Append(@"\"""); break;
					case '\\': es.Append(@"\\" ); break;
					case '\a': es.Append(@"\a" ); break;
					case '\b': es.Append(@"\b" ); break;
					case '\f': es.Append(@"\f" ); break;
					case '\n': es.Append(@"\n" ); break;
					case '\r':                    break;
					case '\t': es.Append(@"\t" ); break;
					case '\v': es.Append(@"\v" ); break;
					default  : es.Append(c     ); break;
				}
			}
			return es.Append('"').ToString();
		}

		public override string UpdateUrl
		{
			get { return @"http://www.exoton.com/downloads/keepass.version"; }
		}
	}
}
